#include <MozziGuts.h>
#include <Oscil.h> 
#include <tables/cos2048_int8.h>
#include <Smooth.h>
#include <AutoMap.h>

// ==== Pin Definitions ====
const int LED1 = 6; // PWM LED reacts to modulation amplitude
const int LED2 = 7; // Blinks in time with modulation
const int LED3 = 8; // Toggles faster with modulation speed

const int BUTTON_PIN = 2;           // Tap Tempo button
const int BUTTON_RANDOM_TEMPO = 3;  // New button D3 for random tempo burst
const int BUTTON_PITCH_DROP = 4;    // New button D4 for pitch drop effect

// Main potentiometers
const int KNOB_PIN1 = A0; // Carrier frequency control
const int KNOB_PIN2 = A1; // Intensity control
const int KNOB_PIN3 = A2; // Modulator ratio base
const int KNOB_PIN4 = A3; 
const int KNOB_PIN5 = A7; 

// Extra LDR sensors
const int LDR_PIN1 = A4; // extra modulation 1
const int LDR_PIN2 = A5; // extra modulation 2
const int LDR_PIN3 = A6; // extra modulation 3

// ==== Mappings ====
const int MIN_CARRIER_FREQ = 20;
const int MAX_CARRIER_FREQ = 200;

const int MIN = 2;
const int MAX = 10;

const int MIN_INTENSITY = 10;
const int MAX_INTENSITY = 100;

AutoMap kMapCarrierFreq(0, 1023, MIN_CARRIER_FREQ, MAX_CARRIER_FREQ);
AutoMap kMapIntensity(0, 1023, MIN_INTENSITY, MAX_INTENSITY);
AutoMap mapThis(0, 1023, MIN, MAX);
AutoMap mapThisToo(0, 1023, MIN, MAX);

// ==== Oscillators ====
Oscil<COS2048_NUM_CELLS, AUDIO_RATE> aCarrier(COS2048_DATA);
Oscil<COS2048_NUM_CELLS, AUDIO_RATE> aModulator(COS2048_DATA);
Oscil<COS2048_NUM_CELLS, CONTROL_RATE> kIntensityMod(COS2048_DATA);

long fm_intensity;
Smooth<long> aSmoothIntensity(0.95f);

// ==== Tap Tempo ====
unsigned long lastTapTime = 0;
unsigned long tapInterval = 500;
unsigned long previousTapInterval = 500;
bool wasPressed = false;

// ==== Random tempo burst state ====
bool randomTempoActive = false;
bool randomTempoButtonWasPressed = false;

// ==== LED Timing ====
unsigned long lastBlinkTime = 0;
bool led2State = false;
bool led3State = false;

// === Variables for pitch drop ===
int carrier_freq_normal = MIN_CARRIER_FREQ;
int carrier_freq_current = MIN_CARRIER_FREQ;

void setup() {
  Serial.begin(115200);
  startMozzi();

  pinMode(BUTTON_PIN, INPUT_PULLUP);
  pinMode(BUTTON_RANDOM_TEMPO, INPUT_PULLUP); // D3 button
  pinMode(BUTTON_PITCH_DROP, INPUT_PULLUP);   // D4 button
  pinMode(LED1, OUTPUT);
  pinMode(LED2, OUTPUT);
  pinMode(LED3, OUTPUT);
}

void updateControl() {
  // ==== TAP TEMPO BUTTON ====
  if (digitalRead(BUTTON_PIN) == LOW) {
    if (!wasPressed) {
      unsigned long now = millis();
      tapInterval = constrain(now - lastTapTime, 100, 2000); // limit range
      lastTapTime = now;
      wasPressed = true;
    }
  } else {
    wasPressed = false;
  }

  // ==== RANDOM TEMPO BURST BUTTON (D3) ====
  bool randomTempoButtonPressed = (digitalRead(BUTTON_RANDOM_TEMPO) == LOW);

  if (randomTempoButtonPressed) {
    if (!randomTempoButtonWasPressed) {
      // Button was just pressed (rising edge)
      randomTempoActive = true;
      previousTapInterval = tapInterval; // save current tempo
      tapInterval = random(100, 1001);   // random tempo between 100ms and 1000ms
    }
  } else {
    // Button released, go back to normal tempo immediately
    if (randomTempoActive) {
      tapInterval = previousTapInterval;
      randomTempoActive = false;
    }
  }
  randomTempoButtonWasPressed = randomTempoButtonPressed;

  // ==== Read main pots ====
  int pot1 = mozziAnalogRead(KNOB_PIN1); // A0
  int pot2 = mozziAnalogRead(KNOB_PIN2); // A1
  int pot3 = mozziAnalogRead(KNOB_PIN3); // A2
  int pot4 = mozziAnalogRead(KNOB_PIN4); // A3
  int pot5 = mozziAnalogRead(KNOB_PIN5); // A7

  // Map main pots as base values
  carrier_freq_normal = kMapCarrierFreq(pot1);
  int intensity_base = kMapIntensity(pot2);
  int modulator_ratio_base = mapThis(pot3);
  int modulator_depth_base = mapThisToo(pot4); // unused but mapped
  int some_other_control = mapThis(pot5); // placeholder, currently unused

  // ==== Read LDR sensors (extra modulation) ====
  int ldr1 = mozziAnalogRead(LDR_PIN1); // A4
  int ldr2 = mozziAnalogRead(LDR_PIN2); // A5
  int ldr3 = mozziAnalogRead(LDR_PIN3); // A6

  // Map LDRs to small modulation values
  int ldr1_mod = map(ldr1, 0, 1023, -10, 10);  // small freq offset
  int ldr2_mod = map(ldr2, 0, 1023, 0, 50);    // intensity scaler
  int ldr3_mod = map(ldr3, 0, 1023, -3, 3);    // modulator ratio offset

  // Combine base values with LDR modulations
  carrier_freq_normal = constrain(carrier_freq_normal + ldr1_mod, MIN_CARRIER_FREQ, MAX_CARRIER_FREQ);
  int intensity = constrain(intensity_base + ldr2_mod, MIN_INTENSITY, MAX_INTENSITY);
  int modulator_ratio = constrain(modulator_ratio_base + ldr3_mod, MIN, MAX);

  // ==== PITCH DROP EFFECT ====
  bool pitchDropPressed = (digitalRead(BUTTON_PITCH_DROP) == LOW);
  if (pitchDropPressed) {
    // Drop pitch by one octave (divide frequency by 2)
    carrier_freq_current = max(carrier_freq_normal / 2, MIN_CARRIER_FREQ);
  } else {
    carrier_freq_current = carrier_freq_normal;
  }

  // Set oscillator frequencies
  aCarrier.setFreq(carrier_freq_current);

  int mod_freq = carrier_freq_current * modulator_ratio;
  aModulator.setFreq(mod_freq);

  // fm_intensity controls modulation amplitude
  fm_intensity = intensity * (kIntensityMod.next() + 128) / 256;

  // ==== Set modulation speed from tap tempo ====
  float mod_speed = 1000.0 / (float)tapInterval;
  kIntensityMod.setFreq(mod_speed);

  // ==== LEDs ====
  // LED1: brightness proportional to fm_intensity
  int brightness = constrain(abs(fm_intensity) / 40, 0, 255);
  analogWrite(LED1, brightness);

  // LED2: blink in tempo with tap button
  if (millis() - lastBlinkTime >= tapInterval) {
    led2State = !led2State;
    digitalWrite(LED2, led2State);
    lastBlinkTime = millis();
  }

  // LED3: faster blinking related to modulation speed
  static unsigned long lastToggleTime = 0;
  int fastBlinkInterval = constrain(tapInterval / 4, 50, 300);
  if (millis() - lastToggleTime >= fastBlinkInterval) {
    led3State = !led3State;
    digitalWrite(LED3, led3State);
    lastToggleTime = millis();
  }
}

int updateAudio() {
  long modulation = aSmoothIntensity.next(fm_intensity) * aModulator.next();
  return aCarrier.phMod(modulation);
}

void loop() {
  audioHook();
}
