#include <MozziGuts.h>
#include <Oscil.h> 
#include <tables/cos2048_int8.h>
#include <Smooth.h>
#include <AutoMap.h>

// ==== Pin Definitions ====
// LEDs
const int LED1 = 6; // PWM LED reacts to modulation amplitude
const int LED2 = 7; // Blinks in time with modulation
const int LED3 = 8; // Toggles faster with modulation speed

// Buttons
const int BUTTON_PIN = 2;           // Tap Tempo button
const int BUTTON_RANDOM_TEMPO = 3;  // Random tempo burst
const int BUTTON_PITCH_DROP = 5;    // Pitch drop effect (optional)
const int BUTTON_OCTAVE_DOWN = 4;   // Octave down effect

// Main potentiometers (Synth controls)
const int KNOB_CARRIER_FREQ = A0;   // Carrier frequency control
const int KNOB_INTENSITY    = A1;   // FM intensity control
const int KNOB_RATIO        = A2;   // Modulator ratio base
const int KNOB_DEPTH        = A3;   // Distortion / bitcrush control

// LDR sensors (extra modulation controls)
const int LDR_FREQ_OFFSET   = A4;   // Carrier frequency offset
const int LDR_INTENSITY     = A5;   // Intensity scaler
const int LDR_RATIO_OFFSET  = A6;   // Modulator ratio offset

// Volume
const int KNOB_VOLUME       = A7;   // Master volume control

// ==== Mappings ====
const int MIN_CARRIER_FREQ = 20;
const int MAX_CARRIER_FREQ = 200;

const int MIN = 2;
const int MAX = 10;

const int MIN_INTENSITY = 10;
const int MAX_INTENSITY = 100;

AutoMap kMapCarrierFreq(1023, 0, MIN_CARRIER_FREQ, MAX_CARRIER_FREQ);
AutoMap kMapIntensity(1023, 0, MIN_INTENSITY, MAX_INTENSITY);
AutoMap mapThis(1023, 0, MIN, MAX);
AutoMap mapThisToo(1023, 0, MIN, MAX); // reversed for distortion pot

// ==== Oscillators ====
Oscil<COS2048_NUM_CELLS, AUDIO_RATE> aCarrier(COS2048_DATA);
Oscil<COS2048_NUM_CELLS, AUDIO_RATE> aModulator(COS2048_DATA);
Oscil<COS2048_NUM_CELLS, CONTROL_RATE> kIntensityMod(COS2048_DATA);

long fm_intensity;
Smooth<long> aSmoothIntensity(0.95f);

// ==== Tap Tempo ====
unsigned long lastTapTime = 0;
unsigned long tapInterval = 500;
unsigned long previousTapInterval = 500;
bool wasPressed = false;

// ==== Random tempo burst state ====
bool randomTempoActive = false;
bool randomTempoButtonWasPressed = false;

// ==== LED Timing ====
unsigned long lastBlinkTime = 0;
bool led2State = false;
bool led3State = false;

// === Variables for pitch/octave ===
int carrier_freq_normal = MIN_CARRIER_FREQ;
int carrier_freq_current = MIN_CARRIER_FREQ;

// === Volume control ===
int volume = 255; // default full volume (0-255 scaling)

// === Distortion control ===
int distortionAmount = 0.5; // from A3 knob

void setup() {
  Serial.begin(115200);
  startMozzi();

  pinMode(BUTTON_PIN, INPUT_PULLUP);
  pinMode(BUTTON_RANDOM_TEMPO, INPUT_PULLUP);
  pinMode(BUTTON_PITCH_DROP, INPUT_PULLUP);
  pinMode(BUTTON_OCTAVE_DOWN, INPUT_PULLUP);
  pinMode(LED1, OUTPUT);
  pinMode(LED2, OUTPUT);
  pinMode(LED3, OUTPUT);
}

void updateControl() {

// ==== TAP TEMPO BUTTON ====
  if (digitalRead(BUTTON_PIN) == LOW) {
    if (!wasPressed) {
      unsigned long now = millis();
      tapInterval = constrain(now - lastTapTime, 100, 2000);
      lastTapTime = now;
      wasPressed = true;
    }
  } else {
    wasPressed = false;
  }

// ==== RANDOM TEMPO BURST BUTTON (D3) ====
  bool randomTempoButtonPressed = (digitalRead(BUTTON_RANDOM_TEMPO) == LOW);

  if (randomTempoButtonPressed) {
    if (!randomTempoButtonWasPressed) {
      randomTempoActive = true;
      previousTapInterval = tapInterval;
      tapInterval = random(100, 1001);
    }
  } else {
    if (randomTempoActive) {
      tapInterval = previousTapInterval;
      randomTempoActive = false;
    }
  }
  randomTempoButtonWasPressed = randomTempoButtonPressed;


//==== Read main pots ====

int potCarrier = mozziAnalogRead(KNOB_CARRIER_FREQ);
int potIntensity = mozziAnalogRead(KNOB_INTENSITY);
int potRatio = mozziAnalogRead(KNOB_RATIO);
int potDepth = mozziAnalogRead(KNOB_DEPTH);
int potVolume = mozziAnalogRead(KNOB_VOLUME);

carrier_freq_normal = kMapCarrierFreq(potCarrier);
int intensity_base = kMapIntensity(potIntensity);
int modulator_ratio_base = mapThis(potRatio);

// Make distortion less intense: scale A3 range down
distortionAmount = mapThisToo(potDepth);        // original mapping
distortionAmount = max(1, distortionAmount / 1.3); // <- scale down by half

volume = map(potVolume, 1023, 0, 0, 255);


// READ LDR SENSORS

  int ldrFreq = mozziAnalogRead(LDR_FREQ_OFFSET);
  int ldrIntensity = mozziAnalogRead(LDR_INTENSITY);
  int ldrRatio = mozziAnalogRead(LDR_RATIO_OFFSET);

  int ldr1_mod = map(ldrFreq, 0, 1023, -10, 10);
  int ldr2_mod = map(ldrIntensity, 0, 1023, 0, 50);
  int ldr3_mod = map(ldrRatio, 0, 1023, -3, 3);

  carrier_freq_normal = constrain(carrier_freq_normal + ldr1_mod, MIN_CARRIER_FREQ, MAX_CARRIER_FREQ);
  int intensity = constrain(intensity_base + ldr2_mod, MIN_INTENSITY, MAX_INTENSITY);
  int modulator_ratio = constrain(modulator_ratio_base + ldr3_mod, MIN, MAX);


// ==== OCTAVE DOWN EFFECT (D5) ====

  bool octaveDownPressed = (digitalRead(BUTTON_OCTAVE_DOWN) == LOW);
  if (octaveDownPressed) {
    carrier_freq_current = max(carrier_freq_normal / 2, MIN_CARRIER_FREQ);
  } else {
    carrier_freq_current = carrier_freq_normal;
  }

  aCarrier.setFreq(carrier_freq_current);
  int mod_freq = carrier_freq_current * modulator_ratio;
  aModulator.setFreq(mod_freq);

  fm_intensity = intensity * (kIntensityMod.next() + 128) / 256;

  float mod_speed = 1000.0 / (float)tapInterval;
  kIntensityMod.setFreq(mod_speed);

  int brightness = constrain(abs(fm_intensity) / 40, 0, 255);
  analogWrite(LED1, brightness);

  if (millis() - lastBlinkTime >= tapInterval) {
    led2State = !led2State;
    digitalWrite(LED2, led2State);
    lastBlinkTime = millis();
  }

  static unsigned long lastToggleTime = 0;
  int fastBlinkInterval = constrain(tapInterval / 4, 50, 300);
  if (millis() - lastToggleTime >= fastBlinkInterval) {
    led3State = !led3State;
    digitalWrite(LED3, led3State);
    lastToggleTime = millis();
  }
}

int updateAudio() {
  long modulation = aSmoothIntensity.next(fm_intensity) * aModulator.next();
  int rawAudio = aCarrier.phMod(modulation);

  // === Apply distortion / bitcrush ===
  int crushed = rawAudio;
  if (distortionAmount > 1) {
    crushed = rawAudio >> (distortionAmount - 1);
    crushed = crushed << (distortionAmount - 1);
  }
  return (crushed * volume) / 255;
}

void loop() {
  audioHook();
}