/*
  Touch pins to MIDI notes

  Plays a MIDI note when a touch pin is touched and modulates it based on the capacitance of that pin.
  Potentiometer on A10 controls octave shift (-2 .. +2). This transposes both BLE-MIDI notes and the local tone.
  (Removed: previous volume control on A10 / CC7)

  Wiring: Pot ends w/ 3V3 & GND, wiper A10.
  Midi Hardware trs output D6

  edit november 2025 *Veerle 
  Code includes:
  - STANDALONE BEEP SOUNDS from onboard output jack (thonkiconn jack)
  - MIDI OVER BLUETOOTH: change your device name if wanted in code
  - MIDI OVER TRS/DIN: if you want TYPE A, keep the pcb silkscreen allignment of the resistors, if you want type B, swap the resistor outputs.

  Bluetooth antenna is included with the kit, this makes the reach much farther to 25 meters or so.
  Measured with no serious measuring device but in footsteps and estimates. 

  by Veerle Pennock & Michelle Vossen(CC BY-NC-SA 4.0)
  https://v0ss3n.github.io/midimadness
*/

#include "pitches.h"
#include <BLEMIDI_Transport.h>
#include <BLEDevice.h>
#include <hardware/BLEMIDI_ESP32.h>
#include "driver/touch_sensor.h"
#include <math.h>

// ---------------- NEW: UART MIDI on D6 ----------------
const int MIDI_TX_PIN = D6;            // hardware MIDI out (DIN-5) TX pin
const uint8_t MIDI_CHANNEL = 1;        // match BLE channel (1..16)

// Use a dedicated hardware serial for MIDI out
HardwareSerial SerialMIDI(1);          // UART1

static inline void HW_MIDI_NoteOn(uint8_t note, uint8_t velocity=127) {
  uint8_t status = 0x90 | ((MIDI_CHANNEL - 1) & 0x0F);
  SerialMIDI.write(status);
  SerialMIDI.write(note & 0x7F);
  SerialMIDI.write(velocity & 0x7F);
}

static inline void HW_MIDI_NoteOff(uint8_t note, uint8_t velocity=0) {
  uint8_t status = 0x80 | ((MIDI_CHANNEL - 1) & 0x0F);
  SerialMIDI.write(status);
  SerialMIDI.write(note & 0x7F);
  SerialMIDI.write(velocity & 0x7F);
}
// ------------------------------------------------------

BLEMIDI_CREATE_INSTANCE("AXELOTLSLOTH", MIDI);
bool isConnected = false;

// pins
int speaker_pin = D7;                                   // piezo / speaker
int pitch_pot_pin = A10;                                // octave pot (-2..+2)
int touch_pins[] = { T1, T2, T3, T4, T5, T6, T7, T8 };  // touch inputs

// touch state
int  touchValues[] = { 0,0,0,0,0,0,0,0 };
int  thresholds[]  = { 100000,100000,100000,100000,100000,100000,100000,100000 };
bool noteActive[]  = { false,false,false,false,false,false,false,false };
int  activeMidiNote[8] = { -1,-1,-1,-1,-1,-1,-1,-1 };
uint32_t lastTouchValues[8] = { 0 };

// pitches + base MIDI notes
int pitches[]   = { NOTE_C5, NOTE_D5, NOTE_E5, NOTE_F5, NOTE_G5, NOTE_A5, NOTE_B5, NOTE_C6 };
int midiNotes[] = { 36, 37, 38, 39, 40, 41, 42, 43 };

// octave shift from A10
int octaveShift = 0;         // -2..+2
int previousOctaveShift = 0;

void setup() {
  Serial.begin(115200);
  pinMode(speaker_pin, OUTPUT);

  // --- NEW: start UART MIDI out on D6 ---
  // RX is unused (-1). 31250 baud, 8N1, no inversion.
  SerialMIDI.begin(31250, SERIAL_8N1, -1, MIDI_TX_PIN);

  // --- Make sure A10 actually behaves as an ADC pin ---
  adcAttachPin(pitch_pot_pin);
  analogSetPinAttenuation(pitch_pot_pin, ADC_11db); // full 0..3.3V range

  MIDI.begin();

  BLEMIDI.setHandleConnected([]() {
    Serial.println("---------CONNECTED---------");
    isConnected = true;
  });
  BLEMIDI.setHandleDisconnected([]() {
    Serial.println("---------NOT CONNECTED---------");
    isConnected = false;
  });
}

static int readPotA10() {
  // Simple 8-sample moving average for stability
  int acc = 0;
  for (int i = 0; i < 8; i++) acc += analogRead(pitch_pot_pin);
  return acc / 8; // ~0..4095
}

void loop() {
  // --- Read & quantize octave from A10 ---
  int octRaw = readPotA10();               // 0..~4095
  int bin    = map(octRaw, 0, 4095, 0, 8); // 0..8
  octaveShift = constrain(bin - 4, -4, 4);

  // Debug so you can confirm the pot moves:
  static uint32_t lastPrint = 0;
  if (millis() - lastPrint > 150) {
    Serial.print("A10 raw: "); Serial.print(octRaw);
    Serial.print("  -> octaveShift: "); Serial.println(octaveShift);
    lastPrint = millis();
  }

  // If the octave changed while notes are held, retune BLE notes cleanly
  if (octaveShift != previousOctaveShift) {
    for (int i = 0; i < 8; i++) {
      if (noteActive[i]) {
        if (activeMidiNote[i] >= 0) {
          // BLE off
          MIDI.sendNoteOff(activeMidiNote[i], 0, 1);
          // HW off
          HW_MIDI_NoteOff(activeMidiNote[i], 0);
        }
        int newNote = constrain(midiNotes[i] + (12 * octaveShift), 0, 127);
        // BLE on
        MIDI.sendNoteOn(newNote, 127, 1);
        // HW on
        HW_MIDI_NoteOn(newNote, 127);

        activeMidiNote[i] = newNote;
      }
    }
    previousOctaveShift = octaveShift;
  }

  int samePins = 0;
  bool isToneActive = false;
  int  totalPitch   = 0;
  int  touchCount   = 0;

  Serial.println("Touch values:");
  for (int i = 0; i < 8; i++) {
    touchValues[i] = touchRead(touch_pins[i]);
    Serial.println(touchValues[i]);

    if (touchValues[i] == lastTouchValues[i]) samePins++;
    lastTouchValues[i] = touchValues[i];

    if (touchValues[i] > thresholds[i]) {
      if (!noteActive[i]) {
        int shiftedNote = constrain(midiNotes[i] + (12 * octaveShift), 0, 127);

        // BLE note on
        MIDI.sendNoteOn(shiftedNote, 127, 1);
        // NEW: hardware UART MIDI note on (D6)
        HW_MIDI_NoteOn(shiftedNote, 127);

        noteActive[i] = true;
        activeMidiNote[i] = shiftedNote;
      }

      int constrainedTouch = constrain(touchValues[i], 150000, 200000);
      int midiModulation   = map(constrainedTouch, 100000, 200000, 0, 127);
      MIDI.sendControlChange(1, midiModulation, 1);  // CC1 modulation (BLE only)

      // Local tone pitch with octave
      float octaveFactor = powf(2.0f, (float)octaveShift);
      int basePitch      = (int)(pitches[i] * octaveFactor);
      int pitchAdjustment= map(constrainedTouch, 100000, 200000, 0, 200);
      int pitch          = basePitch + pitchAdjustment;

      totalPitch += pitch;
      touchCount++;
      isToneActive = true;

    } else {
      if (noteActive[i]) {
        if (activeMidiNote[i] >= 0) {
          // BLE note off
          MIDI.sendNoteOff(activeMidiNote[i], 0, 1);
          // NEW: hardware UART MIDI note off (D6)
          HW_MIDI_NoteOff(activeMidiNote[i], 0);
        }
        noteActive[i] = false;
        activeMidiNote[i] = -1;
      }
    }
  }

  if (!isConnected) {
    if (isToneActive && touchCount > 0) {
      int averagedPitch = totalPitch / touchCount;
      tone(speaker_pin, averagedPitch);
    } else {
      noTone(speaker_pin);
    }
  } else {
    noTone(speaker_pin); // mute speaker if BLE MIDI is connected
  }

  if (samePins == 8) {
    Serial.println("Restarting touch pad...");
    touch_pad_fsm_start();
  }

  delay(10);
}
